/**
 * Async Patterns Evolution Demonstration: Callbacks, Promises, and Async/Await
 * * This file contains the complete step-by-step code demonstrating the evolution
 * of asynchronous handling in JavaScript, from nested callbacks to the modern
 * async/await syntax.
 * * Instructions: Run this file in a Node.js environment or a browser console.
 */

const DELAY = 500;

// ====================================================================
// STEP 1: Define the Mock Asynchronous Operations (using Callbacks)
// ====================================================================

// 1. Simulates fetching user data using a callback
function loadUser(userId, callback) {
  setTimeout(() => {
    console.log(`[Callback-Base] Fetched user ${userId}.`);
    const user = { id: userId, name: "Alice" };
    callback(null, user); // null for no error
  }, DELAY);
}

// 2. Simulates fetching orders using a callback
function loadOrders(user, callback) {
  setTimeout(() => {
    console.log(`[Callback-Base] Fetched orders for ${user.name}.`);
    const orders = ["Laptop", "Monitor"];
    // Introduce a potential error based on user data (for error demo)
    if (user.id === 999) {
        callback(new Error("User is banned!"), null);
    } else {
        callback(null, orders);
    }
  }, DELAY);
}

// ====================================================================
// STEP 2: Demonstrating Callback Hell (The Problem)
// ====================================================================
console.log("\n--- STARTING STEP 2: CALLBACK HELL DEMO ---");
loadUser(123, (userError, user) => {
  if (userError) {
    return console.error("Error loading user:", userError.message);
  }

  // Nested Call 1: Fetches orders after user is loaded
  loadOrders(user, (orderError, orders) => {
    if (orderError) {
      return console.error("Error loading orders:", orderError.message);
    }

    // Final Success Log
    console.log(`\n--- CALLBACK SUCCESS ---`);
    console.log(`Processing Order for: ${user.name}`);
    console.log(`Items: ${orders.join(', ')}`);
  });
});
console.log("--- End of Callback Hell Block (execution continues) ---");


// ====================================================================
// STEP 3: Refactoring to Promises (The Chaining Solution Base)
// ====================================================================

// 1. Returns a Promise for user data
function loadUserPromise(userId) {
  return new Promise((resolve, reject) => {
    setTimeout(() => {
      console.log(`[Promise-Base] Fetched user ${userId}.`);
      const user = { id: userId, name: "Bob" };
      resolve(user); // Success
    }, DELAY);
  });
}

// 2. Returns a Promise for orders
function loadOrdersPromise(user) {
  return new Promise((resolve, reject) => {
    setTimeout(() => {
      console.log(`[Promise-Base] Fetched orders for ${user.name}.`);
      // Simulating rejection (failure)
      if (user.id === 999) {
          reject(new Error("User is banned!")); // Failure
      } else {
          const orders = ["Keyboard", "Mouse"];
          resolve(orders); // Success
      }
    }, DELAY);
  });
}


// ====================================================================
// STEP 4: Chaining Promises for Clean Flow
// ====================================================================
// Note: This promise chain will start executing after the callback timers finish
setTimeout(() => {
    console.log("\n--- STARTING STEP 4: PROMISE CHAIN DEMO ---");
    loadUserPromise(456)
      .then(user => {
        // This .then() handles the user result and returns the next promise
        return loadOrdersPromise(user);
      })
      .then(orders => {
        // This .then() receives the orders data from the previous step
        console.log(`\n--- PROMISE SUCCESS ---`);
        console.log(`Items: ${orders.join(', ')}`);
      })
      .catch(error => {
        // Single error handler for the entire chain!
        console.error("PROMISE CHAIN ERROR:", error.message);
      });
    console.log("--- End of Promise Chain Block (execution continues) ---");
}, DELAY * 4); // Delay execution so it runs after callbacks finish


// ====================================================================
// STEP 5: Introducing Async/Await (The Readability Solution)
// ====================================================================

// Uses the Promise functions from Step 3
async function processOrderModern(userId) {
  // 1. Wrap the entire block in try/catch for synchronous error handling
  try {

    // 2. await pauses here until loadUserPromise resolves
    const user = await loadUserPromise(userId);
    console.log(`[Async/Await] User object ready.`);

    // 3. await pauses again until loadOrdersPromise resolves
    const orders = await loadOrdersPromise(user);

    // 4. Final Success Log
    console.log(`\n--- ASYNC/AWAIT SUCCESS ---`);
    console.log(`Processing Order for: ${user.name}`);
    console.log(`Items: ${orders.join(', ')}`);

  } catch (error) {
    // 5. If any await fails, execution jumps here
    console.error("ASYNC/AWAIT ERROR:", error.message);
  }
}


// ====================================================================
// STEP 6: Final Execution and Comparison
// ====================================================================
setTimeout(() => {
    console.log("\n--- STARTING STEP 6: ASYNC/AWAIT DEMO ---");
    // Calling the modern function (runs asynchronously)
    processOrderModern(789);

    // Example of calling the function that is designed to fail (e.g., user 999)
    // processOrderModern(999);
}, DELAY * 8); // Delay execution so it runs after the promise chain finishes
